<?php

/**
 * Custom walker for the Menu element
 *
 * Class TCB_Menu_Walker
 */
class TCB_Menu_Walker extends Walker_Nav_Menu {
	/**
	 * flag indicating where or not this is a editor page
	 *
	 * @var boolean
	 */
	protected $is_editor_page;

	/**
	 * @var string current menu item
	 */
	protected $current_item_id;

	/**
	 * Stores icon data
	 *
	 * @var array
	 */
	protected $icons = array();

	protected $positional_selectors = false;

	public function __construct() {
		$icons                      = empty( $GLOBALS['tcb_wp_menu']['icon'] ) ? array() : $GLOBALS['tcb_wp_menu']['icon'];
		$this->positional_selectors = tcb_custom_menu_positional_selectors();

		$template = tcb_template( 'elements/menu-item-icon.phtml', null, true, 'backbone' );
		foreach ( (array) $icons as $k => $icon_id ) {
			if ( $icon_id ) {
				$this->icons[ $k ] = str_replace( '_ID_', $icon_id, $template );
			}
		}
	}

	/**
	 * Gets HTML for an icon corresponding to a <li>
	 *
	 * @param WP_Post $item
	 *
	 * @return string
	 */
	protected function icon( $item ) {
		$parent_field = $this->db_fields['parent'];

		/* unlinked id */
		$id = '.menu-item-' . $item->ID;
		if ( $this->positional_selectors && ! empty( $item->_tcb_pos_selector ) ) {
			/* try unlinked positional selectors */
			$id = $item->_tcb_pos_selector;
		}

		if ( isset( $this->icons[ $id ] ) ) {
			return $this->icons[ $id ];
		}

		/* check top level */
		if ( empty( $item->$parent_field ) ) {
			return isset( $this->icons['top'] ) ? $this->icons['top'] : '';
		}

		/**
		 * default : submenu item
		 */
		return isset( $this->icons['sub'] ) ? $this->icons['sub'] : '';
	}

	/**
	 * Display array of elements hierarchically.
	 *
	 * Does not assume any existing order of elements.
	 *
	 * $max_depth = -1 means flatly display every element.
	 * $max_depth = 0 means display all levels.
	 * $max_depth > 0 specifies the number of display levels.
	 *
	 * @since 2.1.0
	 *
	 * @param array $elements  An array of elements.
	 * @param int   $max_depth The maximum hierarchical depth.
	 *
	 * @return string The hierarchical item output.
	 */
	public function walk( $elements, $max_depth ) {
		$args = func_get_args();

		if ( $this->positional_selectors ) {
			$elements  = &$args[0];
			$parent    = $this->db_fields['parent'];
			$top_level = 0;
			$index     = 0;

			/* append some meta-information for each element */
			foreach ( $elements as $menu_item ) {
				if ( empty( $menu_item->$parent ) ) {
					$top_level ++;
					$menu_item->_tcb_index = ++ $index;
				}
			}

			foreach ( $elements as $menu_item ) {
				/**
				 * use positional selectors - in lp-build, these are used to style menu items
				 */
				if ( empty( $menu_item->$parent ) ) {
					if ( 1 === $menu_item->_tcb_index ) {
						$menu_item->_tcb_pos_selector = ':first-child';
					} elseif ( $menu_item->_tcb_index === $top_level ) {
						$menu_item->_tcb_pos_selector = ':last-child';
					} else {
						$menu_item->_tcb_pos_selector = ':nth-child(' . $menu_item->_tcb_index . ')';
					}
				}
			}
		}

		return call_user_func_array( array( $this, 'parent::' . __FUNCTION__ ), $args );
	}

	/**
	 * Starts the list before the elements are added.
	 *
	 * @since 3.0.0
	 *
	 * @see   Walker::start_lvl()
	 *
	 * @param string   $output Used to append additional content (passed by reference).
	 * @param int      $depth  Depth of menu item. Used for padding.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 */
	public function start_lvl( &$output, $depth = 0, $args = array() ) {
		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$indent = str_repeat( $t, $depth );

		// Default class.
		$classes = array( 'sub-menu' );

		/**
		 * Filters the CSS class(es) applied to a menu list element.
		 *
		 * @since 4.8.0
		 *
		 * @param array    $classes The CSS classes that are applied to the menu `<ul>` element.
		 * @param stdClass $args    An object of `wp_nav_menu()` arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 */
		$classes     = apply_filters( 'nav_menu_submenu_css_class', $classes, $args, $depth );
		$classes []  = 'menu-item-' . $this->current_item_id . '-ul';
		$class_names = ' class="' . esc_attr( join( ' ', $classes ) ) . '"';

		$output .= "{$n}{$indent}<ul$class_names>{$n}";
	}

	/**
	 * Starts the element output.
	 *
	 * @since 3.0.0
	 * @since 4.4.0 The {@see 'nav_menu_item_args'} filter was added.
	 *
	 * @see   Walker::start_el()
	 *
	 * @param string   $output Used to append additional content (passed by reference).
	 * @param WP_Post  $item   Menu item data object.
	 * @param int      $depth  Depth of menu item. Used for padding.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 * @param int      $id     Current item ID.
	 */
	public function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
		} else {
			$t = "\t";
		}
		$indent = ( $depth ) ? str_repeat( $t, $depth ) : '';

		$classes = empty( $item->classes ) ? array() : (array) $item->classes;

		/**
		 * Filters the arguments for a single nav menu item.
		 *
		 * @since 4.4.0
		 *
		 * @param stdClass $args  An object of wp_nav_menu() arguments.
		 * @param WP_Post  $item  Menu item data object.
		 * @param int      $depth Depth of menu item. Used for padding.
		 */
		$args = apply_filters( 'nav_menu_item_args', $args, $item, $depth );

		/**
		 * Filters the CSS class(es) applied to a menu item's list item element.
		 *
		 * @since 3.0.0
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param array    $classes The CSS classes that are applied to the menu item's `<li>` element.
		 * @param WP_Post  $item    The current menu item.
		 * @param stdClass $args    An object of wp_nav_menu() arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 */
		$classes = apply_filters( 'nav_menu_css_class', array_filter( $classes ), $item, $args, $depth );
		// make sure these are always included
		$classes[] = 'menu-item-' . $item->ID;
		if ( ! empty( $GLOBALS['tve_menu_font_class'] ) ) {
			$classes[] = $GLOBALS['tve_menu_font_class'];
		}
		if ( $this->is_editor_page() ) {
			$classes[] = 'thrv_wrapper';
		}

		$top_cls = isset( $GLOBALS['tcb_wp_menu']['top_cls'] ) ? (array) $GLOBALS['tcb_wp_menu']['top_cls'] : array();
		if ( 0 === $depth && ! empty( $top_cls ) ) {
			$unlinked_key = ! empty( $item->_tcb_pos_selector ) ? $item->_tcb_pos_selector : '.menu-item-' . $item->ID;
			if ( ! empty( $top_cls[ $unlinked_key ] ) ) {
				$classes [] = $top_cls[ $unlinked_key ];
			} elseif ( ! empty( $top_cls['main'] ) ) {
				$classes [] = $top_cls['main'];
			}
		}

		$class_names = ' class="' . esc_attr( join( ' ', $classes ) ) . '"';

		/**
		 * Filters the ID applied to a menu item's list item element.
		 *
		 * @since 3.0.1
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param string   $menu_id The ID that is applied to the menu item's `<li>` element.
		 * @param WP_Post  $item    The current menu item.
		 * @param stdClass $args    An object of wp_nav_menu() arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 */
		$id = apply_filters( 'nav_menu_item_id', 'menu-item-' . $item->ID, $item, $args, $depth );
		$id = $id ? ' id="' . esc_attr( $id ) . '"' : '';

		$output .= $indent . '<li' . $id . $class_names . ' data-id="' . $item->ID . '">';

		$atts           = array();
		$atts['title']  = ! empty( $item->attr_title ) ? $item->attr_title : '';
		$atts['target'] = ! empty( $item->target ) ? $item->target : '';
		$atts['rel']    = ! empty( $item->xfn ) ? $item->xfn : '';
		$atts['href']   = ! empty( $item->url ) ? $item->url : '';

		/**
		 * Filters the HTML attributes applied to a menu item's anchor element.
		 *
		 * @since 3.6.0
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param array    $atts   {
		 *                         The HTML attributes applied to the menu item's `<a>` element, empty strings are ignored.
		 *
		 * @type string    $title  Title attribute.
		 * @type string    $target Target attribute.
		 * @type string    $rel    The rel attribute.
		 * @type string    $href   The href attribute.
		 * }
		 *
		 * @param WP_Post  $item   The current menu item.
		 * @param stdClass $args   An object of wp_nav_menu() arguments.
		 * @param int      $depth  Depth of menu item. Used for padding.
		 */
		$atts = apply_filters( 'nav_menu_link_attributes', $atts, $item, $args, $depth );

		$attributes = '';
		foreach ( $atts as $attr => $value ) {
			if ( ! empty( $value ) ) {
				$value      = ( 'href' === $attr ) ? esc_url( $value ) : esc_attr( $value );
				$attributes .= ' ' . $attr . '="' . $value . '"';
			}
		}

		/** This filter is documented in wp-includes/post-template.php */
		$title = apply_filters( 'the_title', $item->title, $item->ID );

		/**
		 * Filters a menu item's title.
		 *
		 * @since 4.4.0
		 *
		 * @param string   $title The menu item's title.
		 * @param WP_Post  $item  The current menu item.
		 * @param stdClass $args  An object of wp_nav_menu() arguments.
		 * @param int      $depth Depth of menu item. Used for padding.
		 */
		$title       = apply_filters( 'nav_menu_item_title', $title, $item, $args, $depth );
		$item_output = $args->before;
		$item_output .= '<a' . $attributes . '>' . $this->icon( $item );
		$item_output .= $args->link_before . $title . $args->link_after;
		$item_output .= '</a>';
		$item_output .= $args->after;

		/**
		 * Filters a menu item's starting output.
		 *
		 * The menu item's starting output only includes `$args->before`, the opening `<a>`,
		 * the menu item's title, the closing `</a>`, and `$args->after`. Currently, there is
		 * no filter for modifying the opening and closing `<li>` for a menu item.
		 *
		 * @since 3.0.0
		 *
		 * @param string   $item_output The menu item's starting HTML output.
		 * @param WP_Post  $item        Menu item data object.
		 * @param int      $depth       Depth of menu item. Used for padding.
		 * @param stdClass $args        An object of wp_nav_menu() arguments.
		 */
		$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );

		/* keep a reference to the current menu itemID */
		$this->current_item_id = $item->ID;
	}

	/**
	 * Checks if the current page is the editor page
	 * Also handles the case where the CM is rendered via ajax
	 *
	 * @return boolean
	 */
	protected function is_editor_page() {
		if ( ! isset( $this->is_editor_page ) ) {
			$this->is_editor_page = ( ( wp_doing_ajax() && ! empty( $_REQUEST['action'] ) && $_REQUEST['action'] === 'tcb_editor_ajax' ) || is_editor_page() );
		}

		return $this->is_editor_page;
	}
}
